<?php

/**
 * Tenantos Generic RDNS Provider API Example
 *
 * This example demonstrates how to implement a Generic RDNS API that can be used with
 * the Tenantos Generic RDNS Provider.
 *
 * If you don't receive the Authorization HTTP header, check your webserver configuration.
 * https://www.google.com/search?q=php-fpm+authorization+header+missing
 *
 *
 * **Information about error handling:**
 *
 * - The content of "adminErrorMessage" is shown only to admins. If absent, but "userErrorMessage" is present, the content of "userErrorMessage" will be shown to admins.
 * - The content of "userErrorMessage" is shown to regular users. If absent, a generic error message is used.
 * - In cases where both error types are provided, each audience sees the relevant message.
 * - For HTTP responses outside 200-204, admins receive the raw response for troubleshooting, while users get a generic error message.
 *
 *
 * **Body Data Structure:**
 *
 * The request body contains:
 *
 * - action: The action to perform (testConnection, setIpRecord, getSingleIpRecord, getAllRecordsFromSubnet)
 * - data: Object containing action-specific data (ip, record, subnet, etc.)
 * - meta: Additional context information (see below)
 *
 * **Body Data Enrichment:**
 *
 * This information is normally not needed. To cover edge cases, the body data is enriched to include meta information that aids in providing context to the request. This
 * includes:
 *
 * - meta.user.id: Contains the ID of the currently authenticated user, if any. This helps in associating the request with a specific user.
 * - meta.requestData: Includes the request data (GET, POST parameters, etc.) that the user submitted to Tenantos.
 * - meta.requestDetails: Offers metadata about the HTTP request itself, such as:
 *     - domain: The domain/brand where the request was triggered.
 *     - path: The path of the request, helping in identifying the endpoint being called.
 *     - method: The HTTP method used for the request (e.g., GET, POST).
 *     - ip: The IP address of the Tenantos client.
 *     - url: The full URL of the request.
 * - meta.serverId: The ID of the affected server. Only set if the API is called because the RDNS entry is requested/setted through a server page.
 * Use `file_get_contents('php://input');` to receive the body data, then json_decode the result.
 *
 * Note: If the API call is triggered by a cron job or a background job, the provided meta data may be only partially available.
 */
class RdnsApi {
    private $ptrDatabase = [];

    private const ALLOWED_ACTIONS = [
        'testConnection',
        'setIpRecord',
        'getSingleIpRecord',
        'getAllRecordsFromSubnet',
    ];


    public function __construct() {
        // Example only
        // In production, use a real database or DNS server integration
        $this->ptrDatabase = [
            '12.12.12.1' => 'server1.example.com',
            '12.12.12.2' => 'server2.example.com',
        ];
    }


    public function isAllowedAction($action) {
        return in_array($action, self::ALLOWED_ACTIONS, true);
    }


    /**
     * Test the connection to the RDNS API
     *
     * @return string Returns 'success' if connection is working
     */
    public function testConnection() {
        // Perform any necessary checks (database connection, DNS server availability, etc.)
        // If a different string is returned, the string will be shown on the Tenantos RDNS setup page
        return 'success';
    }


    /**
     * Set or update a PTR record for an IP address
     * All validation is done in the Tenantos backend. You receive a safe input.
     *
     * @param string $ip The IP address
     * @param string $record The PTR record (hostname). Empty string to delete.
     * @return string|null Optional return value
     */
    public function setIpRecord($ip, $record) {
        // Update the PTR record in your backend

        return null;
    }


    /**
     * Get the PTR record for a single IP address
     * All validation is done in the Tenantos backend. You receive a safe input.
     *
     * @param string $ip The IP address
     * @return string The PTR record or empty string if not found
     */
    public function getSingleIpRecord($ip) {
        // Return PTR record or empty string

        return $this->ptrDatabase[$ip] ?? '';
    }


    /**
     * Get all PTR records for a subnet
     * This is only called if the option "Can Manage Unassigned IPs" is called OR if the server has a subnet assignment instead of individual IPs
     *
     * Important: Must return the records quickly. Otherwise, the subnets API gets slow.
     * RDNS records on the subnet manager pages are currently not loaded async.
     *
     * @param string $subnet The subnet (e.g. "10.0.0.0" or "64:ff9b:1::")
     * @param integer $cidr The CIDR
     * @return array Associative array with IP => PTR record
     */
    public function getAllRecordsFromSubnet($subnet, $cidr) {
        // Example: Filter records by subnet
        // In production, this should query your database/API for all IPs in the subnet
        $result = [];
        $subnetPrefix = implode('.', array_slice(explode('.', $subnet), 0, 3));

        foreach ($this->ptrDatabase as $ip => $record) {
            if (strpos($ip, $subnetPrefix) === 0) {
                $result[$ip] = $record;
            }
        }

        return $result;
    }


    public static function sendError(array $errorDetails) {
        http_response_code(400);
        echo json_encode($errorDetails);
        exit;
    }
}


$headers = getallheaders();

if (!isset($headers['Authorization']) || $headers['Authorization'] !== 'YourSecretBearerToken') {
    RdnsApi::sendError([
        'adminErrorMessage' => 'Access token is invalid or missing.',
        'userErrorMessage' => 'Generic error'
    ]);
}


$payload = file_get_contents('php://input');
$body = json_decode($payload, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    RdnsApi::sendError([
        'adminErrorMessage' => 'Invalid JSON in request body: ' . json_last_error_msg(),
        'userErrorMessage' => 'Generic error'
    ]);
}

$action = $body['action'] ?? '';
$data = $body['data'] ?? [];
$meta = $body['meta'] ?? []; // Optional meta data

$api = new RdnsApi();

if ($api->isAllowedAction($action)) {
    $result = null;

    switch ($action) {
        case 'testConnection':
            $result = $api->testConnection();
            break;

        case 'setIpRecord':
            $result = $api->setIpRecord($data['ip'], $data['record']);
            break;

        case 'getSingleIpRecord':
            $result = $api->getSingleIpRecord($data['ip']);
            break;

        case 'getAllRecordsFromSubnet':
            $result = $api->getAllRecordsFromSubnet($data['subnet'], $data['cidr']);
            break;

        default:
            RdnsApi::sendError([
                'adminErrorMessage' => "Unknown action: {$action}",
                'userErrorMessage' => 'Invalid action specified.'
            ]);
    }

    echo json_encode([
        'result' => $result
    ]);
} else {
    RdnsApi::sendError([
        'adminErrorMessage' => "Method does not exist: {$action}",
        'userErrorMessage' => 'Invalid action specified.'
    ]);
}
